;**************************************************************************************
;*
;* sxprog.asm
;*
;* (c) Ben Stragnell 1/14/99
;*
;**************************************************************************************
;*
;* PIC16F84 code for programming a Scenix SX chip or another PIC.
;*
;* Freq: 10MHz
;*
;*   B0 <-> SX OSC2 (or Target PIC RB6 [clock])
;*   B1  -> PC Parallel Port
;*   B2  -> PC Parallel Port
;*   B3 <-  PC Parallel Port D4
;*   B4 <-  PC Parallel Port D0
;*   B5 <-  PC Parallel Port D1
;*   B6 <-  PC Parallel Port D2
;*   B7 <-  PC Parallel Port D3
;*   A0  -> SX OSC1 Voltage Control (5V enable)
;*   A1  -> SX OSC1 Voltage Control (12V enable)
;*   A2  -> PC Parallel Port
;*   A3  -> PC Parallel Port
;*   A4  -> Target PIC RB7 [data] via 4.7K pullup
;* MCLR <-  PC Parallel Port D5
;*
;**************************************************************************************

;*
;* device and config options
;*

		LIST		P=16F84

#include <p16f84.inc>

		__config	h'3FFA'

;*
;* definitions
;*

;*
;* registers
;*

DATALO		equ		h'10'
DATAHI		equ		h'11'
BITCT		equ		h'12'
VALID		equ		h'13'
REPTCT		equ		h'14'
ICTR		equ		h'15'
TDATALO		equ		h'16'
TDATAHI		equ		h'17'
FROMSXLO	equ		h'18'
FROMSXHI	equ		h'19'
ERRORS		equ		h'1a'
PCCMD		equ		h'1b'
TMP		equ		h'1c'
SER_OUT		equ		h'1d'
SER_IN		equ		h'1e'
SERMODE		equ		h'1f'

;*
;* PC commands
;*

PCCMD_NOP	equ		h'00'
PCCMD_OSC1	equ		h'01'
PCCMD_OSC2	equ		h'02'
PCCMD_ECHO	equ		h'03'
PCCMD_SX	equ		h'04'
PCCMD_PICINIT	equ		h'05'
PCCMD_PICCMD	equ		h'06'
PCCMD_PICCMDD	equ		h'07'
PCCMD_PICCLOSE	equ		h'08'
PCCMD_RESET	equ		h'09'
PCCMD_PICPROG	equ		h'0a'

;*
;* bit assignments for PORTB and PORTA
;*

; PORTB
BIT_OSC2	equ		0
BIT_PCFIRST	equ		1
BIT_PCREQ	equ		2
BIT_PCACK	equ		3
BIT_PCDI0	equ		4
BIT_PCDI1	equ		5
BIT_PCDI2	equ		6
BIT_PCDI3	equ		7

; PORTA
BIT_OSC1_5V	equ		0
BIT_OSC1_12V	equ		1
BIT_PCDO0	equ		2
BIT_PCDO1	equ		3

;*
;* error codes
;*

ERROR_OK		equ	d'0'
ERROR_CMDINVALID	equ	d'1'
ERROR_BITCT		equ	d'2'
ERROR_TIMEOUT		equ	d'3'

;*
;* code
;*

		org		0

;**************************************************************************************
;*
;* start/intvec
;*
;**************************************************************************************
;*
;* CPU startup/interrupt entry pts
;*
;**************************************************************************************

start		clrf		INTCON
		goto		main
		nop
		nop

;**************************************************************************************
;*
;* interrupt handler... we're catching a falling edge on B0
;*
;**************************************************************************************

intvec		; we have roughly 25 clocks before the data should be placed
		; (1.5 128Kcycles=29 clocks - 4 for int latency).
		movlw		d'7'			; 4
		movwf		ICTR			; 5
delaylp1	decfsz		ICTR,1			; 6..
		goto		delaylp1		; 25

		; fetch data, and place it
		rlf		DATALO,1		; 26
		rlf		DATAHI,1		; 27
		bsf		STATUS,RP0		; 28
		bsf		TRISB,BIT_OSC2		; 29
		btfss		STATUS,C		; 30
		bcf		TRISB,BIT_OSC2		; 31 *
		bcf		STATUS,RP0		; 32

		; ensure data is filled with 1's
		bsf		DATALO,0		; 33

		; wait until 38 to sample data
		nop					; 34
		rlf		FROMSXLO,1		; 35
		rlf		FROMSXHI,1		; 36
		bcf		FROMSXLO,0		; 37
		btfsc		PORTB,BIT_OSC2		; 38 *
		bsf		FROMSXLO,0		; 39

		; wait until around 49 to remove our data
		incf		BITCT,1			; 40
		goto		dummya1			; 41-42
dummya1		goto		dummya2			; 43-44
dummya2		goto		dummya3			; 45-46
dummya3		bsf		STATUS,RP0		; 47
		nop					; 48
		bsf		TRISB,BIT_OSC2		; 49 *
		bcf		STATUS,RP0		; 50

		; the next edge will either arrive at around 78 or, if the
		; next edge is a sync, at around 156. we split the difference,
		; and decide at 117. enable the interrupts, and see if we
		; fall through.

		movlw		b'10010000'		; 51
		movwf		INTCON			; 52
		movlw		d'21'			; 53
		movwf		ICTR			; 54
delaylp2	decfsz		ICTR,1			; 55-116
		goto		delaylp2

		; ok... we fell through - we're in a sync pulse
		; we can expect the next edge to arrive at around 156, so we
		; have around 39 clocks to do the end-of-word processing

		; see if we saw the correct number of bits...
		movlw		d'16'
		subwf		BITCT,1			; clear BITCT at same time!
		btfsc		STATUS,Z
		goto		bitct_ok

		; invalid bit count - did we expect it to be correct?
		clrf		BITCT
		btfsc		VALID,0
		goto		bitct_error		; doh!
		goto		wait			; no worries - wait for next pulse

		; we received the correct number of bits in the frame
bitct_ok
		; now, were we actually sending?
		btfss		VALID,0
		goto		not_sending

		; do we have more frames to transmit?
		decf		REPTCT,1
		btfsc		STATUS,Z
		goto		cmd_ok

not_sending:
		; we expect subsequent frames to be correct
		bsf		VALID,0

		; copy data - note that we only start sending once we've seen
		; completely valid frame go by...
		movf		TDATALO,0
		movwf		DATALO
		movf		TDATAHI,0
		movwf		DATAHI

		; await next interrupt
		; we'll wait for at least a whole frame before declaring a timeout

wait		clrf		ICTR
delaylp5	nop
		nop
		nop
		decfsz		ICTR,1
		goto		delaylp5

		; ok... if we got here, the SX chip just isn't sending
		goto		timeout


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

main
		;*
		;* Initialise IO ports
		;*

		clrf		PORTB			; port B is low when output enabled
		bsf		STATUS,RP0		; select bank 1
		movlw		b'11111001'
		movwf		TRISB			; B0=input!
		movlw		b'11110000'		; A=all outputs except A4
		movwf		TRISA

		;*
		;* Initialise TMR0
		;*

		movlw		b'00001000'		; no prescale on tmr0, pullups ENABLED
							; interrupt on B0 falling edge
		movwf		OPTION_REG		; write into option register
		bcf		STATUS,RP0		; select bank 0 again

		;*
		;* delay for a short while
		;*

;		goto		serial_test

		clrf		PORTA
		clrf		ICTR
lp1		clrf		BITCT
lp2		nop
		nop
		nop
		decfsz		BITCT,1
		goto		lp2
		decfsz		ICTR,1
		goto		lp1

		;*
		;* everything is fine to start with
		;*

		clrf		ERRORS
		clrf		FROMSXLO
		clrf		FROMSXHI

		;*
		;* come up in parallel mode
		;*

		clrf		SERMODE

		;*
		;* the main loop involves receiving a 32-bit command from the PC
		;* while sending a 16-bit response back. If an SX command is issued,
		;* the appropriate registers are loaded, and the interrupt routine
		;* is allowed to take over. It will return via either timeout,
		;* bitct_error, or cmd_ok.
		;*

mainlp
		; build data to transmit  EEEE:HHHH:LLLL:LLLL
		movlw		b'00001111'
		andwf		FROMSXHI,1
		rlf		ERRORS,1
		rlf		ERRORS,1
		rlf		ERRORS,1
		rlf		ERRORS,0
		andlw		b'11110000'
		iorwf		FROMSXHI,1

		; are we in serial or parallel mode?
		btfsc		SERMODE,0		; serial mode?
		goto		serxchg

		; parallel mode
		call		xchg_pc			; send ERRORS/FROMSX, receive PCCMD/REPTCT/TDATA
		goto		continue

		; serial mode
serxchg		call		xchg_pc_ser

continue
		; clear errors and response for next time
		clrf		ERRORS
		clrf		FROMSXLO
		clrf		FROMSXHI

		; decode command
		movf		PCCMD,0
		sublw		PCCMD_NOP		; OSC1 voltage level command?
		btfsc		STATUS,Z
		goto		mainlp

		movf		PCCMD,0
		sublw		PCCMD_OSC1		; OSC1 voltage level command?
		btfsc		STATUS,Z
		goto		cmd_osc1

		movf		PCCMD,0
		sublw		PCCMD_OSC2		; OSC2 low/tristate command?
		btfsc		STATUS,Z
		goto		cmd_osc2

		movf		PCCMD,0
		sublw		PCCMD_ECHO		; echo command?
		btfsc		STATUS,Z
		goto		cmd_echo

		movf		PCCMD,0
		sublw		PCCMD_SX		; SX command?
		btfsc		STATUS,Z
		goto		cmd_sx

		movf		PCCMD,0
		sublw		PCCMD_PICINIT		; Init PIC for programming?
		btfsc		STATUS,Z
		goto		cmd_picinit

		movf		PCCMD,0
		sublw		PCCMD_PICCMD		; Do a PIC command?
		btfsc		STATUS,Z
		goto		cmd_piccmd

		movf		PCCMD,0
		sublw		PCCMD_PICCMDD		; Do a PIC command with data?
		btfsc		STATUS,Z
		goto		cmd_piccmd

		movf		PCCMD,0
		sublw		PCCMD_PICCLOSE		; Finish programming PIC?
		btfsc		STATUS,Z
		goto		cmd_picclose

		movf		PCCMD,0
		sublw		PCCMD_RESET		; reset programmer?
		btfsc		STATUS,Z
		goto		start

		movf		PCCMD,0
		sublw		PCCMD_PICPROG		; begin/end programming
		btfsc		STATUS,Z		; (for 16C64's ?)
		goto		cmd_picprog

		goto		cmd_invalid

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_osc1	movf		TDATALO,0
		andlw		b'00000011'
		movwf		PORTA
		goto		mainlp

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_osc2	bcf		PORTB,0
		bsf		STATUS,RP0
		btfsc		TDATALO,0
		goto		cmd_osc2_set
		bcf		PORTB,0
		goto		cmd_osc2_done
cmd_osc2_set	bsf		PORTB,0
cmd_osc2_done	bcf		STATUS,RP0
		bcf		PORTB,0
		goto		mainlp

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_echo	movf		TDATALO,0
		movwf		FROMSXLO
		movf		TDATAHI,0
		movwf		FROMSXHI
		goto		mainlp

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_invalid	movlw		ERROR_CMDINVALID
		movwf		ERRORS
		goto		mainlp

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_sx		; clear flags
		clrf		VALID
		clrf		BITCT

		; ensure B0 is LOW when not tristate
		bcf		PORTB,0

		; set data to a safe value - all 1's initially
		movlw		h'FF'
		movwf		DATAHI
		movwf		DATALO

		; allow the interrupt processing to take over
		movlw		b'10010000'
		movwf		INTCON

		goto		wait


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

bitct_error	; disable further interrupts
		clrf		INTCON

		; set error code
		movlw		ERROR_BITCT
		movwf		ERRORS

		; continue
		goto		mainlp

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

timeout		; disable further interrupts
		clrf		INTCON

		; set error code
		movlw		ERROR_TIMEOUT
		movwf		ERRORS

		; continue
		goto		mainlp

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_ok		; disable further interrupts
		clrf		INTCON

		; set error code
		clrf		ERRORS

		; continue
		goto		mainlp

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

xchg_pc

		bsf		PORTB,BIT_PCFIRST	; signal first packet

		; wait for REQ==ACK --> PC ready to receive
		clrf		VALID			; reset serial detect counter
		call		xchg_wait

		; set loop ctr - 8 packets
		movlw		d'8'
		movwf		BITCT

xchg_lp
		; put 2 data bits onto PORTA
		rlf		FROMSXLO,1
		rlf		FROMSXHI,1
		rlf		TMP,1
		rlf		FROMSXLO,1
		rlf		FROMSXHI,1
		rlf		TMP,1
		rlf		TMP,1
		rlf		TMP,0
		andlw		b'00001100'
		movwf		TMP
		movf		PORTA,0
		andlw		b'11110011'
		iorwf		TMP,0
		movwf		PORTA

		; toggle REQ
		movlw		(1<<BIT_PCREQ)
		xorwf		PORTB,1

		; wait for PC to respond
		clrf		VALID			; reset serial detect ctr
		call		xchg_wait

		; receive 4 data bits from PORTB
		movf		PORTB,0
		movwf		TMP

		movlw		d'4'
		movwf		ICTR
recv_lp		rlf		TMP,1
		rlf		TDATALO,1
		rlf		TDATAHI,1
		rlf		REPTCT,1
		rlf		PCCMD,1
		decfsz		ICTR,1
		goto		recv_lp

		; clear FIRST
		bcf		PORTB,BIT_PCFIRST

		; rinse and repeat
		decfsz		BITCT,1
		goto		xchg_lp

		; paranoia - ensure we didn't break B0 through the tristate gotcha...
		; note - this is OK with the PIC stuff, since B0 is low by default...
		bcf		PORTB,0

		; our work here is done
		return


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

		; wait for REQ==ACK
xchg_wait
		rlf		PORTB,0			; get PORTB<<1
		xorwf		PORTB,0			; get (PORTB<<1)^PORTB
		movwf		TMP
		btfss		TMP,BIT_PCACK		; REQ==ACK?
		return					; if so, return

		; now, has B4 changed? if so, we may be receiving serial data
		movf		PORTB,0			; read port B
		andlw		b'00010000'		; mask off relevant bit
		movwf		SER_OUT			; store
		xorwf		SER_IN,0		; xor with last value
		btfsc		STATUS,Z		; no change?
		goto		xchg_wait		; if Z, continue waiting

		movf		SER_OUT,0		; update value
		movwf		SER_IN

		decfsz		VALID,1			; dec ctr...
		goto		xchg_wait		; repeat until 256 transitions

		; ok... we have now received 256 transitions on B4, with no
		; transitions on PCACK... this would suggest that the PC is trying
		; to talk to us in serial mode.

		bsf		SERMODE,0		; set serial flag

		movlw		d'8'			; send !!!!!!!! to tell the PC to be
		movwf		ICTR			; quiet
shutup_lp	movlw		'!'
		movwf		SER_OUT
		call		serial_send
		clrf		TMP
pause		decfsz		TMP,1
		goto		pause
		decfsz		ICTR,1
		goto		shutup_lp

		goto		mainlp			; jump straight back into main loop


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_piccmd	; B0 and A4 should both be OUTPUT/LOW at this stage...
		; we're going to clock 6 bytes of command data from REPTCT, then
		; clock 16 bytes of data from TDATAHI/LO - in the case of read commands,
		; A4 will be tristate anyway (handy pullup).

		movlw		d'6'
		movwf		ICTR

pic_cmdlp1
		bsf		PORTB,0			; raise clock
		bsf		STATUS,RP0		; access second page
		bsf		TRISA,4			; tristate by default
		rrf		REPTCT,1		; get next bit in carry
		btfss		STATUS,C		; is it clear?
		bcf		TRISA,4			; if so, clear data.
		bcf		STATUS,RP0		; access first page
		bcf		PORTB,0			; lower clock, PIC latches data
		decfsz		ICTR,1			; dec ctr
		goto		pic_cmdlp1		; loop if not zero

		movf		PCCMD,0			; check PC command - data required?
		sublw		PCCMD_PICCMDD
		btfss		STATUS,Z
		goto		done_shortcmd

		bsf		STATUS,RP0		; access second page
		bsf		TRISA,4			; set data tristate... by default
		bcf		STATUS,RP0		; access first page

		movlw		d'16'
		movwf		ICTR

pic_cmdlp2	bsf		PORTB,0			; raise clock
		bsf		STATUS,RP0		; access second page
		bsf		TRISA,4			; tristate by default
		rrf		TDATAHI,1		; rotate data...
		rrf		TDATALO,1		; get next bit in carry
		btfss		STATUS,C		; is it clear?
		bcf		TRISA,4			; if so, clear bit
		bcf		STATUS,RP0		; access first page
		bcf		PORTB,0			; lower clock, PIC latches data
		rrf		FROMSXHI,1		; read data ourselves
		rrf		FROMSXLO,1
		bcf		FROMSXHI,7
		btfsc		PORTA,4
		bsf		FROMSXHI,7

		decfsz		ICTR,1			; dec ctr
		goto		pic_cmdlp2		; loop if not zero

		nop					; ensure PIC has read data
		nop					; (if applicable)
		nop
		nop

		bsf		STATUS,RP0		; access second page
		bcf		TRISA,4			; set A4 to output
		bcf		STATUS,RP0		; access first page

		; take a copy of FROMSXHI into ERRORS, so it won't get overwritten
		; when building the PIC's response to the PC
		rrf		FROMSXHI,0
		movwf		ERRORS
		rrf		ERRORS,1
		rrf		ERRORS,1
		rrf		ERRORS,1

		; return to main loop
		goto		mainlp

done_shortcmd
		bsf		STATUS,RP0		; access second page
		bcf		TRISA,4			; ensure data is output!
		bcf		STATUS,RP0		; access first page
		goto		mainlp


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_picprog	; B0 and A4 should both be OUTPUT/LOW at this stage...
		; we're going to clock 6 bits of command data from REPTCT,
		; then wait for around 250 clocks (100us).
		; then we're going to clock 6 bits of command data from TDATAHI

		movlw		d'6'
		movwf		ICTR
pic_proglp1
		bsf		PORTB,0			; raise clock
		bsf		STATUS,RP0		; access second page
		bsf		TRISA,4			; tristate by default
		rrf		REPTCT,1		; get next bit in carry
		btfss		STATUS,C		; is it clear?
		bcf		TRISA,4			; if so, clear data.
		bcf		STATUS,RP0		; access first page
		bcf		PORTB,0			; lower clock, PIC latches data
		decfsz		ICTR,1			; dec ctr
		goto		pic_proglp1		; loop if not zero

		bsf		STATUS,RP0		; access second page
		bsf		TRISA,4			; set data tristate... by default
		bcf		STATUS,RP0		; access first page

		movlw		d'82'			; delay for 250 clocks
		call		delay

		movlw		d'6'
		movwf		ICTR

pic_proglp2	bsf		PORTB,0			; raise clock
		bsf		STATUS,RP0		; access second page
		bsf		TRISA,4			; tristate by default
		rrf		TDATAHI,1		; rotate data...
		btfss		STATUS,C		; is it clear?
		bcf		TRISA,4			; if so, clear bit
		bcf		STATUS,RP0		; access first page
		bcf		PORTB,0			; lower clock, PIC latches data
		decfsz		ICTR,1			; dec ctr
		goto		pic_proglp2		; loop if not zero

		bsf		STATUS,RP0		; access second page
		bcf		TRISA,4			; set A4 to output
		bcf		STATUS,RP0		; access first page

		; return to main loop
		goto		mainlp


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_picinit	; lower A4 and B0 - ensure both are outputs
		bsf		STATUS,RP0		; access second page
		movlw		b'11100000'		; all A outputs
		movwf		TRISA
		movlw		b'11111000'		; B0-2 outputs
		movwf		TRISB
		bcf		STATUS,RP0		; access first page
		bcf		PORTB,0			; lower bits
		bcf		PORTA,4
		goto		mainlp			; done

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

cmd_picclose	; return A4 and B0 to tristate
		bsf		STATUS,RP0		; access second page
		movlw		b'11110000'		; A4 now input
		movwf		TRISA
		movlw		b'11111001'		; B0 now input
		movwf		TRISB
		bcf		STATUS,RP0		; access first page
		bcf		PORTB,0			; ensure bits are low
		bcf		PORTA,4
		goto		mainlp			; done

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

serial_test

		call		serial_recv
		btfss		ERRORS,0
		goto		gotone

		movlw		'!';
		movwf		SER_OUT
		call		serial_send

		goto		serial_test

gotone		movf		SER_IN,0
		movwf		SER_OUT
		call		serial_send
		goto		serial_test


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

serial_send	bcf		PORTA,3

		movlw		d'9'			; set up loop ctr
		movwf		BITCT
		goto		ser_clear		; start with START bit

ser_lp		rrf		SER_OUT,1		; 61
		btfsc		STATUS,C		; 62
		goto		ser_set			; 63..64
		nop
ser_clear	bcf		PORTA,2			; 0
		goto		ser_next		; 1..2
ser_set		bsf		PORTA,2			; 0
		goto		ser_next		; 1..2
ser_next	movlw		d'16'			; 3
		call		delay			; 4..55
		nop					; 56
		nop					; 57
		decfsz		BITCT,1			; 58
		goto		ser_lp			; 59..60
		nop					; 61
		nop					; 62
		nop					; 63
		nop					; 64
		bsf		PORTA,2			; 0
		bsf		PORTA,3
		movlw		d'19'			; 1
		call		delay			; 2..62
		return


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

serial_recv
		; wait for B4 low, approx 0.13s timeout
		clrf		ICTR			; timeout ctr
rtlp1		clrf		TMP			; ctr2
rtlp2		btfss		PORTB,4			; low?
		goto		gotlow			; yeehaw!
		decfsz		TMP,1
		goto		rtlp2
		decfsz		ICTR,1
		goto		rtlp1

		; timeout - return error
recv_err	movlw		d'1'
		movwf		ERRORS
		return

gotlow		; got a start bit? assume we caught it at 2... wait to 1/2 bit
		movlw		d'7'			; 3
		call		delay			; 4..28
		nop					; 29
		nop					; 30
		movlw		d'8'			; 31
		movwf		BITCT			; 32
		btfsc		PORTB,4			; 33... ensure still low
		goto		recv_err		; 34

		movlw		d'18'			; 35
		call		delay			; 36..93
		nop					; 94

ser_rlp		; receive bit
		rrf		SER_IN,1		; 63 [95]
		bcf		SER_IN,7		; 64 [96]
		btfsc		PORTB,4			; 0  [97]
		bsf		SER_IN,7		; 1
		nop					; 2
		nop					; 3
		movlw		d'17'			; 4
		call		delay			; 5..59
		decfsz		BITCT,1			; 60
		goto		ser_rlp			; 61..62

		; ensure stop bit is set
		btfss		PORTB,4
		goto		recv_err

		; delay 1/2 bit(ish)
		movlw		d'7'
		call		delay
		clrf		ERRORS
		return


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;* delay for 3n+4 (including call/return)
;*
;**************************************************************************************

delay		movwf		TMP
delaylp		decfsz		TMP,1
		goto		delaylp
		return


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

xchg_pc_ser	movlw		'*'
		movwf		SER_OUT
		call		serial_send

ret1		call		serial_recv
		btfsc		ERRORS,0
		goto		ret1
		movf		SER_IN,0
		movwf		PCCMD

		movf		FROMSXHI,0
		movwf		SER_OUT
		call		serial_send

ret2		call		serial_recv
		btfsc		ERRORS,0
		goto		ret2
		movf		SER_IN,0
		movwf		REPTCT

		movf		FROMSXLO,0
		movwf		SER_OUT
		call		serial_send

ret3		call		serial_recv
		btfsc		ERRORS,0
		goto		ret3
		movf		SER_IN,0
		movwf		TDATAHI

		movf		FROMSXLO,0
		xorlw		d'255'
		xorwf		FROMSXHI,0
		movwf		SER_OUT
		call		serial_send

ret4		call		serial_recv
		btfsc		ERRORS,0
		goto		ret4
		movf		SER_IN,0
		movwf		TDATALO

		return


;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************

;**************************************************************************************
;*
;*
;*
;**************************************************************************************
;*
;*
;*
;**************************************************************************************



;*
;* data
;*


;*
;* end
;*


		end
