/***********************************************************************
 *
 * fluffy.c
 *
 ***********************************************************************
 *
 * new PC software for fluffy PIC/SX programmer (28th sept!)
 *
 ***********************************************************************/

/*
 * includes
 */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>

/*
 * defines
 */

// chip definition flags
#define FLG_FLASH			1
#define FLG_SX				2
#define FLG_SXMANGLE			4

// SX ISP commands
#define SX_ERASE	0x0
#define SX_READDEVICE	0x1
#define SX_READFUSEX	0x2
#define SX_PROGFUSEX	0x3
#define SX_LOADDATA	0x4
#define SX_PROGDATA	0x5
#define SX_READDATA	0x6
#define SX_INC		0x7
#define SX_NOP		0xF

// PIC ISP commands (for EPROM chips)
#define PIC_LOADCFG		0x000000
#define PIC_LOADDATA		0x020000
#define PIC_READDATA		0x040000
#define PIC_INCADDR		0x060000
#define PIC_BEGINPROG		0x080000
#define PIC_ENDPROG		0x0E0000

// PIC ISP commands (for FLASH chips)
#define PIC_BEGINPROGONLY	0x180000
#define PIC_LOADE2DATA		0x030000
#define PIC_READE2DATA		0x050000
#define PIC_BULKERASEPROG	0x090000
#define PIC_BULKERASEE2		0x0B0000


// parallel port bits
#define DATAOUT_MASK	15
#define ACK_MASK	16
#define MCLR_MASK	32
#define DATAIN_SHIFT	3
#define DATAIN_MASK	24
#define REQ_MASK	32
#define FIRST_MASK	64

// timeout
#define TIMEOUT		100
#define RETRY_LIMIT	25

// target cmds
#define TARGET_CMD_NOP		0x00000000
#define TARGET_CMD_OSC1		0x01000000
#define TARGET_CMD_OSC2		0x02000000
#define TARGET_CMD_ECHO		0x03000000
#define TARGET_CMD_SX		0x04000000
#define TARGET_CMD_PICINIT	0x05000000
#define TARGET_CMD_PICCMD	0x06000000
#define TARGET_CMD_PICCMDD	0x07000000
#define TARGET_CMD_PICCLOSE	0x08000000
#define TARGET_CMD_RESET	0x09000000
#define TARGET_CMD_PICPROG	0x0a000000

/*
 * typedefs
 */

typedef unsigned long	uint32;
typedef unsigned short	uint16;
typedef unsigned char	uint8;
typedef unsigned char	byte;
typedef signed long	int32;
typedef signed short	int16;
typedef signed char	int8;

#define MAX_CODESIZE	8192
#define MAX_EEPROMSIZE	1024
#define MAX_USERSIZE	1024



/*
 * structs
 */

typedef struct _chip
{
	char		*name;

	int32		codeSize;
	int32		eepromSize;
	int32		userSize;

	int32		codeOrg;
	int32		eepromOrg;
	int32		userOrg;

	uint32		flags;

	int32		cpMask;
	int32		writeRept;
} chip;

/*
 * globals
 */


// pic definitions
chip			chips[]=
{
	{
		"16F84",
		1024,		64,		8,
		0x0000,		0x2100,		0x2000,
		FLG_FLASH,
		0x0010,
		0,
	},
	{
		"16C84",
		1024,		64,		8,
		0x0000,		0x2100,		0x2000,
		FLG_FLASH,
		0x3FF0,
		0,
	},
	{
		"16C64",
		2048,		0,		8,
		0x0000,		0x0000,		0x2000,
		0,
		0x0030,
		0,
	},
	{
		"SX1828OLD",
		2048,		0,		32,
		0x0000,		0x0000,		0x1000,
		FLG_SX | FLG_SXMANGLE,
		0x0008,
		100
	},
	{
		"SX1828NEW",
		2048,		0,		32,
		0x0000,		0x0000,		0x1000,
		FLG_SX,
		0x0008,
		100
	},
};

// memory read from .hex file
int32			codeMem[MAX_CODESIZE];
int32			eepromMem[MAX_EEPROMSIZE];
int32			userMem[MAX_USERSIZE];

// current chip
chip			*cc;

// hex file name
char			hexFile[256];

// command-line overrides for fuse config
int32			cmdFuse,cmdFusex;
int32			progFuse,progFusex;
int32			curFuse,curFusex,curDevice;

// various flags
int32			comPort;
int32			readMode;

// write stats
int32			actualWrites;

// hardware stuff
int32			basePort;
int32			portOut;

// misc buffers
char			tmpString[256];
char			tmpString2[256];

/*
 * function prototypes
 */

void			Delay(int32 p);
int32			SerOpen(void);
int32			SerRecv(void);
void			SerSend(int32 ch);
int32			ParseArgs(int32 argc,char **argv);
int32			HexRead(char *fn);
int32			ParOpen(void);
int32			TargetSwap(uint32 out,int32 *in);
int32			TargetOpen(void);
int32			TargetCmd(uint32 out,int32 *in);

int32			PICOpen(void);
int32			PICClose(void);
int32			PICProcess(void);

int32			SXOpen(void);
int32			SXClose(void);
int32			SXProcess(void);
int32			SXCmd(int32 rept,int32 cmd,int32 data,int32 *result);

/*
 * code
 */


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void main(int argc, char **argv)
{
	int32		i,j,k;

	printf("\n\nfluffy - a SX/PIC programmer\n");
	printf("____________________________________________________________________________\n\n");

	// clear out memory
	memset(codeMem,-1,sizeof(codeMem));
	memset(userMem,-1,sizeof(userMem));
	memset(eepromMem,-1,sizeof(eepromMem));

	// check cmd line args
	i=ParseArgs(argc,argv);
	if(i)
	{
		ShowUsage();
		exit(1);
	}

	// read hex file
	if(!readMode)
	{
		i=HexRead(hexFile);
		if(i)
		{
			printf("error reading hex file\n");
			exit(1);
	 	}
	}

	// initialise hardware port
	if(comPort)	i=SerOpen();
	else		i=ParOpen();
	if(i)
	{
		printf("error opening hardware port\n");
		exit(1);
	}

	// initialise target
	i=TargetOpen();
	if(i)
	{
		printf("error opening target\n");
		exit(1);
	}

	// print out current config
	printf("\n");
	printf("chip    : %s\n",cc->name);
	if(comPort)	printf("port    : serial   [COM%d]\n",comPort);
	else		printf("port    : parallel [LPT1]\n");
	printf("\n");

	if(cc->flags & FLG_SX)		i=SXProcess();
	else				i=PICProcess();
	if(i)
	{
		printf("fatal error\n");
		exit(1);
	}
	else
	{
		printf("\ndone\n");
		exit(0);
	}
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void ShowUsage(void)
{
	int		i;

	printf("usage: fluffy input.hex\n");
	printf("              -com1\n");
	printf("              -com2\n");
	printf("              -com3\n");
	printf("              -com4\n");
	printf("              -fuse XXX   (eg... -fuse 40B)\n");
	printf("              -fusex XXX   (eg... -fusex B7E)\n");
	printf("              -read\n");
	printf("              -chip <chipname>\n");
	printf("                 supported chips:\n");
	for(i=0;i<sizeof(chips)/sizeof(chip);i++)
	{
		printf("                     %s\n",chips[i].name);
	}
	printf("\n");
	printf("eg:  fluffy simple.sxh -com2 -chip sx1828old\n");
}


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 ParseArgs(int32 argc,char **argv)
{
	int32		i,j,k;

	// set default values
	comPort=0;
	cmdFuse=-1;
	cmdFusex=-1;
	readMode=0;
	cc=NULL;
	hexFile[0]=0;

	for(k=1;k<argc;k++)
	{
		if(argv[k][0]=='-')
		{
			if(!stricmp(argv[k],"-fuse"))
			{
				i=sscanf(argv[k+1],"%lx",&cmdFuse);
				if(i!=1)
				{
					printf("invalid hex value: %s\n",argv[k+1]);
					return(1);
				}
				k++;
			}
			else if(!stricmp(argv[k],"-fusex"))
			{
				i=sscanf(argv[k+1],"%lx",&cmdFusex);
				if(i!=1)
				{
					printf("invalid hex value: %s\n",argv[k+1]);
					return(1);
				}
				k++;
			}
			else if(!stricmp(argv[k],"-read"))
			{
				readMode=1;
			}
			else if(!stricmp(argv[k],"-chip"))
			{
				for(i=0;i<sizeof(chips)/sizeof(chip);i++)
				{
					if(!stricmp(chips[i].name,argv[k+1]))
					{
						cc=&chips[i];
					}
				}
				if(!cc)
				{
					printf("unrecognised chip name: %s\n",argv[k+1]);
					return(1);
				}
				k++;
			}
			else if(toupper(argv[k][1]=='c') &&
				toupper(argv[k][2]=='o') &&
				toupper(argv[k][3]=='m') &&
				strlen(argv[k])==5)
			{
				comPort=argv[k][4]-'0';
				if(comPort<1 || comPort>4)
				{
					printf("invalid com port selection\n");
					return(1);
				}
			}
			else
			{
				printf("unrecognised command line option: %s\n",argv[k]);
				return(1);
			}
		}
		else
		{
			if(hexFile[0])
			{
				printf("too many filenames specified\n");
				return(1);
			}
			strcpy(hexFile,argv[k]);
		}
	}

	// got a chip?
	if(!cc)
	{
		printf("no target chip specified\n");
		return(1);
	}

	// got a hex file?
	if(!hexFile[0] && !readMode)
	{
		printf("no hex file specified\n");
		exit(1);
	}

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 HexRead4(char *s)
{
	int32		a;
	a=s[0]; a=toupper(a); a-='0'; if(a>9) a-=7;
	return(a);
}

int32 HexRead8(char *s)
{
	int32		a;
	a =HexRead4(s) << 4;
	a|=HexRead4(s+1);
	return(a);
}

int32 HexRead16(char *s)
{
	int32		a;
	a =HexRead8(s) << 8;
	a|=HexRead8(s+2);
	return(a);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 HexRead(char *fn)
{
	FILE		*stream;
	int32		i,j,k;
	int32		bc,addr,type;

	// open file
	stream=fopen(fn,"r");
	if(!stream)	return(1);

	// read it
	while(1)
	{
		// read line
		fgets(tmpString,255,stream);
		if(feof(stream)) break;

		// strip CR
		if(tmpString[strlen(tmpString)-1]=='\n')
			tmpString[strlen(tmpString)-1]=0;

		// check 1st char
		if(tmpString[0]!=':')	return(2);

		// read byte count, address and record type
		bc=HexRead8(&tmpString[1]);
		addr=HexRead16(&tmpString[3]);
		type=HexRead8(&tmpString[7]);

		// if type is 1, we're done
		if(type==1) break;

		// type not 0?
		if(type!=0)
		{
			printf("warning: ignoring unrecognised type field: $%02lx\n",type);
			continue;
		}

		// read bytes
		for(i=0;i<bc;i+=2)
		{
			j=HexRead16(&tmpString[9+(i*2)]);
			j=((j>>8)&0x00FF) | ((j<<8)&0xFF00);
			k=(addr+i)>>1;

			// decide where to write...
			if(k>=cc->codeOrg && k<(cc->codeOrg+cc->codeSize))
			{
				codeMem[k-cc->codeOrg]=j;
			}
			else if(k>=cc->eepromOrg && k<(cc->eepromOrg+cc->eepromSize))
			{
				eepromMem[k-cc->eepromOrg]=j;
			}
			else if(k>=cc->userOrg && k<(cc->userOrg+cc->userSize))
			{
				userMem[k-cc->userOrg]=j;
			}
			else
			{
				printf("warning: ignoring out-of-range data at : $%04lx\n",k);
			}

		}
	}

	// close file
	fclose(stream);
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PICOpen(void)
{
	int32		i,j,k,cmd;

	// take /MCLR to 0V
	cmd=TARGET_CMD_OSC1 | 0x0000;
	i=TargetCmd(cmd,&j);
	if(i || j!=0) return(1);

	Delay(25);

	// lower CLOCK/DATA
	cmd=TARGET_CMD_PICINIT;
	i=TargetCmd(cmd,&j);
	if(i || j!=0) return(1);

	Delay(25);

	// go to 12V
	cmd=TARGET_CMD_OSC1 | 0x0002;
	i=TargetCmd(cmd,&j);
	if(i || j!=0) return(1);

	Delay(25);

	// hopefully ok...
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PICClose(void)
{
	int32		i,j,k,cmd;

	// take /MCLR to 0V
	cmd=TARGET_CMD_OSC1 | 0x0000;
	i=TargetCmd(cmd,&j);
	if(i || j) return(1);

	// restore CLOCK/DATA to tristate
	cmd=TARGET_CMD_PICCLOSE;
	i=TargetCmd(cmd,&j);
	if(i || j!=0) return(2);

	// go to 5V
	cmd=TARGET_CMD_OSC1 | 0x0001;
	i=TargetCmd(cmd,&j);
	if(i || j!=0) return(3);

	// hopefully ok...
	return(0);
}


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PICDump(int32 size,int32 cmd)
{
	int32		i,j,k;
	int32		addr;

	for(addr=0;addr<size;addr+=8)
	{
		printf("%04lx : ",addr);
		for(k=0;k<8;k++)
		{
			i=TargetCmd(TARGET_CMD_PICCMDD | cmd | 0xFFFF,&j);
			if(i) { printf("error!\n"); PICClose(); return(1); }

			printf("%04lx ",(j>>1)&0x3fff);

			i=TargetCmd(TARGET_CMD_PICCMD | PIC_INCADDR | 0xFFFF,&j);
			if(i) { printf("error!\n"); PICClose(); return(1); }
		}
		printf("\n");
	}
	printf("\n");

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PICWrite(int32 val,int32 readCmd,int32 loadCmd,int32 limit)
{
	int32		i,j,k;
	int32		retry,r;

	// prepare val
	k=(val & 0x3fff) << 1;

	// try writing
	retry=0;
	while(retry<limit)
	{
		// read current data
		i=TargetCmd(TARGET_CMD_PICCMDD | readCmd | 0xFFFF,&r);
		if(i) return(i);

		r=(r>>1) & 0x3fff; r<<=1;

		// already same?
		if(r==k) break;

		// need to write - load up data
		i=TargetCmd(TARGET_CMD_PICCMDD | loadCmd | k,&j);
		if(i) return(i);

		// write method varies depending on flash or eeprom
		if(cc->flags & FLG_FLASH)
		{
			i=TargetCmd(TARGET_CMD_PICCMD | PIC_BEGINPROG | 0xFFFF,&j);
			if(i) return(i);
		}
		else
		{
			i=TargetCmd(TARGET_CMD_PICPROG | PIC_BEGINPROG | (PIC_ENDPROG>>8) | 0xFF,&j);
			if(i) return(i);
		}
		actualWrites++;

		// small delay
		Delay(10);

		// inc retry counter
		retry++;
	}

	// failed?
	if(retry==limit)
	{
		printf("\n failure: %04lx %04lx\n",k>>1,r>>1);
		return(1);
	}

	// groovy
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PICProcess(void)
{
	int32		i,j,k,l;
	int32		addr;

	// connect to target pic
	i=PICOpen();
	if(i) return(1);

	// fetch config word
	printf("reading configuration word... "); fflush(stdout);

	// cmd 0
	i=TargetCmd(TARGET_CMD_PICCMDD | 0xFFFF,&j);
	if(i) { printf("error!\n"); PICClose(); return(1); }

	// advance ptr
	for(k=0;k<7;k++)
	{
		i=TargetCmd(TARGET_CMD_PICCMD | PIC_INCADDR | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }
	}

	// read config word
	i=TargetCmd(TARGET_CMD_PICCMDD | PIC_READDATA | 0xFFFF,&j);
	if(i) { printf("error!\n"); PICClose(); return(1); }
	curFuse=(j>>1) & 0x3fff;
	printf("%04lx\n",curFuse);

	// reset target PIC
	PICClose(); PICOpen();

	// read?
	if(readMode)
	{
		// check code protect bits
		if((curFuse & cc->cpMask)!=cc->cpMask)
		{
			printf("\none or more code-protect bits are set - unable to read device\n");
			PICClose();
			return(1);
		}

		// read code
		printf("\n\nprogram data:\n");
		i=PICDump(cc->codeSize,PIC_READDATA);
		if(i) { printf("error!\n"); PICClose(); return(1); }
		PICClose(); PICOpen();

		// read user data
		printf("\n\nuser data:\n");
		i=TargetCmd(TARGET_CMD_PICCMDD | PIC_LOADCFG | 0xFFFF,&j);
		i=PICDump(cc->userSize,PIC_READDATA);
		if(i) { printf("error!\n"); PICClose(); return(1); }
		PICClose(); PICOpen();

		// read eeprom data
		if(cc->eepromSize)
		{
			printf("\n\neeprom data:\n");
			i=PICDump(cc->eepromSize,PIC_READE2DATA);
			if(i) { printf("error!\n"); PICClose(); return(1); }
			PICClose(); PICOpen();
		}

		// done
		PICClose();
		return(0);
	}

	// override fuse?
	if(cmdFuse!=-1)		userMem[0x7]=cmdFuse & 0x3fff;

	// check we have a suitable fuse
	cmdFuse=userMem[0x7];
	if(cmdFuse==-1)
	{
		printf("error - no configuration word present on command line or in HEX file\n");
		return(1);
	}


	// check code protect bits
	if((curFuse & cc->cpMask)!=cc->cpMask)
	{
		// eprom devices can only be UV erased
		if(!(cc->flags & FLG_FLASH))
		{
			printf("device is code-protected - unable to write!\n");
			PICClose();
			return(1);
		}

		// need to disable CP
		printf("device is code-protected - disabling code-protect... "); fflush(stdout);

		i=TargetCmd(TARGET_CMD_PICCMDD | PIC_LOADCFG | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }

		for(k=0;k<7;k++)
		{
			i=TargetCmd(TARGET_CMD_PICCMD | PIC_INCADDR | 0xFFFF,&j);
			if(i) { printf("error!\n"); PICClose(); return(1); }
		}

		i=TargetCmd(TARGET_CMD_PICCMD | 0x010000 | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }
		i=TargetCmd(TARGET_CMD_PICCMD | 0x070000 | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }

		i=TargetCmd(TARGET_CMD_PICCMD | 0x080000 | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }

		i=clock(); while(clock()-i < 10);

		i=TargetCmd(TARGET_CMD_PICCMD | 0x010000 | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }
		i=TargetCmd(TARGET_CMD_PICCMD | 0x070000 | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }

		i=clock(); while(clock()-i < 10);

		PICClose(); PICOpen();
		printf("ok\n");
	}

	// write code
	actualWrites=0;
	printf("writing code "); fflush(stdout);
	for(addr=0;addr<cc->codeSize;addr++)
	{
		i=PICWrite(codeMem[addr],PIC_READDATA,PIC_LOADDATA,RETRY_LIMIT);
		if(i) { printf("error at %04lx!\n",addr); PICClose(); return(1); }
		i=TargetCmd(TARGET_CMD_PICCMD | PIC_INCADDR | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }
		printf("."); fflush(stdout);
	}
	printf("\n");
	PICClose(); PICOpen();

	// write eeprom memory
	if(cc->eepromSize)
	{
		printf("writing eeprom data "); fflush(stdout);
		for(addr=0;addr<cc->eepromSize;addr++)
		{
			i=PICWrite(eepromMem[addr],PIC_READE2DATA,PIC_LOADE2DATA,RETRY_LIMIT);
			if(i) { printf("error!\n"); PICClose(); return(1); }
			i=TargetCmd(TARGET_CMD_PICCMD | PIC_INCADDR | 0xFFFF,&j);
			if(i) { printf("error!\n"); PICClose(); return(1); }
			printf("."); fflush(stdout);
		}
		printf("\n");
		PICClose(); PICOpen();
	}

	// write user
	printf("writing user data "); fflush(stdout);

	i=TargetCmd(TARGET_CMD_PICCMDD | PIC_LOADCFG | 0xFFFF,&j);
	if(i) { printf("error!\n"); PICClose(); return(1); }
	for(addr=0;addr<cc->userSize;addr++)
	{
		if(addr<4 || addr>6)
		{
			i=PICWrite(userMem[addr],PIC_READDATA,PIC_LOADDATA,200);
			if(i) { printf("error!\n"); PICClose(); return(1); }
		}
		i=TargetCmd(TARGET_CMD_PICCMD | PIC_INCADDR | 0xFFFF,&j);
		if(i) { printf("error!\n"); PICClose(); return(1); }
		printf("."); fflush(stdout);
	}
	printf("\n");

	printf("\ndone - [%d actual writes performed]\n",actualWrites);

	// disconnect from target pic
	i=PICClose();
	if(i) return(1);

	// groovy
	return(0);
}


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXOpen(void)
{
	int32		i,j,k;

	// take /MCLR to 0V
	i=TargetCmd(TARGET_CMD_OSC1 | 0x0000,&j);
	if(i || j!=0) return(1);

	Delay(25);

	// lower OSC2
	i=TargetCmd(TARGET_CMD_OSC2 | 0x0000,&j);
	if(i || j!=0) return(1);

	Delay(25);

	// toggle OSC1 from 0V to 5V [9 times]
	for(k=0;k<10;k++)
	{
		i=TargetCmd(TARGET_CMD_OSC1 | 0x0001,&j);
		if(i) { printf("error\n"); return(1); }
		Delay(5);
		i=TargetCmd(TARGET_CMD_OSC1 | 0x0000,&j);
		if(i) { printf("error\n"); return(1); }
		Delay(5);
	}

	// wait a moment... hopefully long enough for 9 internal clocks
	Delay(200);

	// release OSC2
	i=TargetCmd(TARGET_CMD_OSC2 | 0x0001,&j);
	if(i) { printf("error\n"); return(1); }

	Delay(100);

	// raise OSC1 to 12V
	i=TargetCmd(TARGET_CMD_OSC1 | 0x0002,&j);
	if(i) { printf("error\n"); return(1); }

	Delay(500);

	// see if the PIC can detect a 128KHz ISP signal
	for(k=0;k<10;k++)
	{
		Delay(100);
		i=SXCmd(1,SX_NOP,0xFFF,&j);
		if(i<0) { printf("error\n"); return(1); }
		if(!i) break;
		printf("."); fflush(stdout);
	}
	if(j!=0x0fff)
	{
		i=TargetCmd(TARGET_CMD_OSC1 | 0x0000,&j);
		return(2);
	}
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXClose(void)
{
	int32		i,j,k;

	// exit ISP mode
	i=TargetCmd(TARGET_CMD_OSC1 | 0x0000,&j);
	if(i) { printf("error\n"); return(1); }
	Delay(100);

	i=TargetCmd(TARGET_CMD_OSC1 | 0x0001,&j);
	if(i) { printf("error\n"); return(1); }
	Delay(100);

	i=TargetCmd(TARGET_CMD_OSC1 | 0x0000,&j);
	if(i) { printf("error\n"); return(1); }
	Delay(100);

	// ensure no ISP signal
	i=SXCmd(1,SX_NOP,0xFFF,&j);
	if(i!=3)
	{
		printf("\nSX still seems to be in ISP mode!\n");
		return(2);
	}

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXCmd(int32 rept,int32 cmd,int32 data,int32 *result)
{
	uint32		q;
	int32		i,j;

	q =(uint32)TARGET_CMD_SX;
	q|=((uint32)rept)<<16;
	q|=((uint32)cmd)<<12;
	q|=((uint32)data) & 0xFFF;

	i=TargetCmd(q,&j);
	if(i) return(-1);

	*result=j & 0xFFF;

	return((j>>12) & 0xF);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXWrite(int32 *data,int len,int rept,int retry)
{
	int32		i,j,k,l,r;

	for(k=0;k<len;k++)
	{
		l=data[k] & 0xfff;
		if(!(k&15)) { printf("."); fflush(stdout); }
		if(l!=0xfff)
		{
			printf("."); fflush(stdout);
			for(r=0;r<retry;r++)
			{
				i=SXCmd(1,SX_LOADDATA,l,&j);
				if(i) return(1);

				i=SXCmd(rept,SX_PROGDATA,0xFFF,&j);
				if(i) return(1);

				i=SXCmd(rept,SX_READDATA,0xFFF,&j);
				if(i) return(1);

				if(j==l)	break;
			}
			if(r==retry)	return(1);
		}

		// advance data ptr
		i=SXCmd(1,SX_INC,0xFFF,&j);
		if(i) return(1);
	}

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXProcess(void)
{
	int32		i,j,k,l;

	// did we get fuse/fusex setting on cmd line?
	if(cmdFuse!=-1)		progFuse=cmdFuse;
	else			progFuse=userMem[0x10];
	if(cmdFusex!=-1)	progFusex=cmdFusex;
	else			progFusex=userMem[0x11];

	// check we got fuse/fusex settings from somewhere...
	if(!readMode)
	{
		if(progFuse==-1)
		{
			printf("error - no FUSE word present on command line or in HEX file\n");
			return(1);
		}
		if(progFusex==-1)
		{
			printf("error - no FUSEX word present on command line or in HEX file\n");
			return(1);
		}
	}

	printf("initialising SX chip..."); fflush(stdout);
	i=SXOpen();
	if(i)
	{
		printf("error\n");
		return(1);
	}
	printf("ok\n");
	printf("reading configuration...\n\n");

	i=SXCmd(1,SX_READFUSEX,0xFFF,&curFusex);
	if(i) {	printf("error\n"); SXClose(); return(1); }

	i=SXCmd(1,SX_READDATA,0xFFF,&curFuse);
	if(i) {	printf("error\n"); SXClose(); return(1); }

	// preserve top bits?
	if(cc->flags & FLG_SXMANGLE)
	{
		progFusex = (progFusex & 0x07F) | (curFusex & 0xF80);
		if(progFusex & 0x020)	progFusex |= 0x010;
		else			progFusex &= ~0x010;
	}

	// check code protection!
	if((curFuse & cc->cpMask)!=cc->cpMask)
	{
		if(readMode)
		{
			printf("device is code protected - unable to read\n");
			return(1);
		}

		printf("disabling code-protection for writing..."); fflush(stdout);

		// write non-CP'd fuse register
		l=(progFuse | cc->cpMask) & 0xfff;

		i=SXCmd(0,SX_ERASE,progFusex,&j);
		if(i) return(1);
		i=SXCmd(1,SX_LOADDATA,l,&j);
		if(i) return(1);
		i=SXCmd(cc->writeRept,SX_PROGDATA,0xFFF,&j);
		if(i) return(1);

		// reset
		i=SXClose(); if(i) { printf("error closing!\n"); return(1); }
		i=SXOpen(); if(i) { printf("error opening!\n"); return(1); }

		// read fuse and fusex again
		i=SXCmd(1,SX_READDATA,0xFFF,&curFuse);
		if(i) {	printf("error\n"); SXClose(); return(1); }
		i=SXCmd(1,SX_READFUSEX,0xFFF,&curFusex);
		if(i) {	printf("error\n"); SXClose(); return(1); }


		printf("ok\n");
	}

	i=SXCmd(1,SX_READDEVICE,0xFFF,&curDevice);
	if(i) {	printf("error\n"); SXClose(); return(1); }


	printf("current device config:\n");
	printf("  Device          : [%03lx]\n",curDevice);
	printf("  Fusex           : [%03lx]\n",curFusex);
	printf("  Fuse            : [%03lx]\n",curFuse);
	printf("\n");

	// read?
	if(readMode)
	{
		printf("program data:\n");
		for(i=0;i<cc->codeSize;i+=8)
		{
			printf("%04x : ",i);
			for(j=0;j<8;j++)
			{
				k=SXCmd(1,SX_INC,0xFFF,&l);
				if(k) {	printf("error\n"); SXClose(); return(1); }
				k=SXCmd(1,SX_READDATA,0xFFF,&l);
				if(k) {	printf("error\n"); SXClose(); return(1); }

				printf("%03x ",l);
			}
			printf("\n");
		}
		printf("\n\n");

		// done
		SXClose();
		return(0);
	}

	// ok... time to program that sucker...
	printf("erasing... "); fflush(stdout);
	i=SXCmd(0,SX_ERASE,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); return(1); }
	printf("done\n");

	// write fusex
	printf("writing fusex [%03lx]...",progFusex); fflush(stdout);
	i=SXCmd(1,SX_LOADDATA,progFusex,&j);
	if(i) {	printf("error %d\n",i); SXClose(); return(1); }
	i=SXCmd(cc->writeRept,SX_PROGFUSEX,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); return(1); }
	printf("ok\n");

	// write fuse
	printf("writing fuse  [%03lx]...",progFuse); fflush(stdout);
	i=SXWrite(&progFuse,1,cc->writeRept,5);
	if(i) { printf("write error\n",i); SXClose(); return(1); }
	printf("ok\n");

	// write code
	printf("writing code"); fflush(stdout);
	i=SXWrite(codeMem,cc->codeSize,cc->writeRept,5);
	if(i) { printf("write error\n",i); SXClose(); return(1); }
	printf("ok\n");

	// close device
	printf("\nclosing SX chip..."); fflush(stdout);
	i=SXClose();
	if(i)
	{
		printf("error closing SX chip\n");
		return(1);
	}

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SerOpen(void)
{
	unsigned short		*p;
	unsigned short far	*fp;

	// get address - deal with real/prot-mode stuff
	if(sizeof(fp)==6)
	{
		p=(unsigned short *)0x00000400;
		basePort=p[comPort-1];
	}
	else
	{
		fp=(unsigned short far *)0x00000400;
		basePort=fp[comPort-1];
	}
	if(!basePort) return(1);
	printf("using serial uart at %03x\n",basePort);

	// disable port int32errupts
	outp(basePort+1,0);

	// disable FIFO mode
	outp(basePort+2,0);

	outp(basePort+4,0);

	// set 38400 baud, select 8-bit words
	outp(basePort+3,128);
	outp(basePort+0,0x03);
	outp(basePort+1,0x00);
	outp(basePort+3,3);		// 8-bits

	return(0);
}


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SerRecv(void)
{
	int32	j;

	if(!(inp(basePort+5)&1)) return(-1);
	j=inp(basePort);
	return(j);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void SerSend(int32 ch)
{
	while(!(inp(basePort+5)&32));
	outp(basePort,ch);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 ParOpen(void)
{
	// use LPT1
	basePort=888;
	outp(basePort+2,0);
	outp(basePort,0);
	portOut=0;

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void ParWrite(int32 mask,int32 bits)
{
	portOut &= ~mask;
	portOut |= (bits & mask);
	outp(basePort,portOut);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 ParRead(void)
{
	return(inp(basePort+1));
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 TargetSwap(uint32 out,int32 *in)
{
	int32		i,j,k,l,m;

	if(!comPort)
	{
		// ensure FIRST is high
		for(i=0;i<TIMEOUT;i++)
		{
			j=ParRead();
			if(j & FIRST_MASK) break;
			Delay(1);
			break;
		}
		if(i==TIMEOUT) return(1);

		// exchange 8 blocks
		m=0;
		for(k=0;k<8;k++)
		{
			// wait for REQ to be ~ACK
			for(i=0;i<TIMEOUT;i++)
			{
				j=ParRead();
				if( (j&REQ_MASK) && !(portOut&ACK_MASK) ) break;
				if( !(j&REQ_MASK) && (portOut&ACK_MASK) ) break;
				Delay(1);
			}
			if(i==TIMEOUT) return(1);

			// receive 2 bits
			l=(j & DATAIN_MASK) >> DATAIN_SHIFT;
			m<<=2;
			m|=l;

			// place our data on the bus
			ParWrite(DATAOUT_MASK,(out>>28) & DATAOUT_MASK);

			// shift data
			out <<= 4;

			// toggle ACK
			ParWrite(ACK_MASK,~(portOut & ACK_MASK));
		}
	}
	else
	{
		// serial port mode... assume ready to go
		while((j=SerRecv())!='*');
		SerSend((out>>24)&0xFF);
		while((j=SerRecv())==-1);
		m=j<<8;
		l=j ^ 0xff;
		SerSend((out>>16)&0xFF);
		while((j=SerRecv())==-1);
		m|=j;
		l^=j;
		SerSend((out>>8)&0xFF);
		while((j=SerRecv())==-1);
		if(l!=j)
		{
			printf("chksum error\n"); return(2);
		}
		SerSend((out>>0)&0xFF);
	}

	// write back result
	*in=m;

	return(0);
}


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 TargetOpen(void)
{
	int32		i,j,k;

	printf("resetting target... "); fflush(stdout);

	if(!comPort)
	{
		// lower MCLR
		ParWrite(MCLR_MASK,0);

		// wait a moment
		Delay(10);

		// raise MCLR
		ParWrite(MCLR_MASK,MCLR_MASK);
	}
	else
	{
		// send 0x09 (reset) until we get a '!'
		for(i=0;i<1000;i++)
		{
			SerSend(0x09);
			k=-1;
			for(j=0;j<1000;j++)
			{
				k=SerRecv();
				if(k!=-1) break;
			}
			if(k=='!') break;
		}
		if(i==1000)
		{
			return(2);
		}

		// wait for '*'
		while(SerRecv()!='*');

		// send nop
		SerSend(0);
		while(SerRecv()==-1);	// hi
		SerSend(0);
		while(SerRecv()==-1);	// lo
		SerSend(0);
		while(SerRecv()==-1);	// chk
		SerSend(0);
	}

	// try a NOP
	i=TargetSwap(0x00000000,&j);
	if(i)
	{
		printf("failed on NOP\n");
		return(1);
	}

	// try an ECHO
	i=TargetCmd(0x03001249,&j);
	if(j!=0x0249)
	{
		printf("failed on ECHO\n");
		return(i);
	}
	printf("ok\n");

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 TargetCmd(uint32 out,int32 *in)
{
	int32		i;

	// send cmd
	i=TargetSwap(out,in);
	if(i) return(1);

	// get reply
	i=TargetSwap(0x00000000,in);
	if(i) return(1);

	// cool
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void Delay(int32 p)
{
	int32		i;
	if(comPort)
		for(i=0;i<p*1000;i++) inp(basePort+5);
	else
		for(i=0;i<p*1000;i++) inp(basePort+1);
}




