/***********************************************************************
 *
 * fluffy.c
 *
 ***********************************************************************
 *
 * PC software for cheap SX programmer
 *
 * Ben Stragnell
 *
 ***********************************************************************/

/*
 * includes
 */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

/*
 * defines
 */

#define SX_ERASE	0x0
#define SX_READDEVICE	0x1
#define SX_READFUSEX	0x2
#define SX_PROGFUSEX	0x3
#define SX_LOADDATA	0x4
#define SX_PROGDATA	0x5
#define SX_READDATA	0x6
#define SX_INC		0x7
#define SX_NOP		0xF

#define DATAOUT_MASK	15
#define ACK_MASK	16
#define MCLR_MASK	32

#define DATAIN_SHIFT	3
#define DATAIN_MASK	24
#define REQ_MASK	32
#define FIRST_MASK	64

#define TIMEOUT		1000000

#define PROGRAM_SIZE    2048
#define PICPROGRAM_SIZE	1024
#define USER_SIZE	8
#define EEPROM_SIZE	64

/*
 * typedefs
 */

typedef signed long	int32;
typedef unsigned long	uint32;
typedef signed short	int16;
typedef unsigned short	uint16;
typedef unsigned char	byte;

/*
 * structs
 */

/*
 * globals
 */

int32		forceFusex=0;

int32		writeRept=100;
int32		verify=0;
int32		readMode=0;
int32		picMode=0;
int32		progFuse=-1;
int32		progFusex=-1;
int32		comPort=0;

int		testMode=0;

char		inputFile[1024];
char		tmpString[1024];
char		tmpString2[1024];


int32		basePort;
int32		portOut;

int32		sxDevice;
int32		sxFuse;
int32		sxFusex;

unsigned short	progData[PROGRAM_SIZE];
unsigned short	userData[USER_SIZE];
unsigned short	eepromData[EEPROM_SIZE];

/*
 * function prototypes
 */

void		Delay(int32);
int32		SerRecv(void);

/*
 * code
 */


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void ParInit(int32 port)
{
	basePort=port;
	outp(basePort+2,0);
	outp(basePort,0);
	portOut=0;
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void ParWrite(int32 mask,int32 bits)
{
	portOut &= ~mask;
	portOut |= (bits & mask);
	outp(basePort,portOut);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 ParRead(void)
{
	return(inp(basePort+1));
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PicSwap(uint32 out,int32 *in)
{
	int32		i,j,k,l,m;

	if(!comPort)
	{
		// ensure FIRST is high
		for(i=0;i<TIMEOUT;i++)
		{
			j=ParRead();
			if(j & FIRST_MASK) break;
			break;
		}
		if(i==TIMEOUT) return(1);

		// exchange 8 blocks
		m=0;
		for(k=0;k<8;k++)
		{
			// wait for REQ to be ~ACK
			for(i=0;i<TIMEOUT;i++)
			{
				j=ParRead();

				if( (j&REQ_MASK) && !(portOut&ACK_MASK) ) break;
				if( !(j&REQ_MASK) && (portOut&ACK_MASK) ) break;
			}
			if(i==TIMEOUT) return(1);

			// receive 2 bits
			l=(j & DATAIN_MASK) >> DATAIN_SHIFT;
			m<<=2;
			m|=l;

			// place our data on the bus
			ParWrite(DATAOUT_MASK,(out>>28) & DATAOUT_MASK);

			// shift data
			out <<= 4;

			// toggle ACK
			ParWrite(ACK_MASK,~(portOut & ACK_MASK));
		}
	}
	else
	{
		// serial port mode... assume ready to go
		while((j=SerRecv())!='*');
		SerSend((out>>24)&0xFF);
		while((j=SerRecv())==-1);
		m=j<<8;
		l=j ^ 0xff;
		SerSend((out>>16)&0xFF);
		while((j=SerRecv())==-1);
		m|=j;
		l^=j;
		SerSend((out>>8)&0xFF);
		while((j=SerRecv())==-1);
		if(l!=j)
		{
			printf("chksum error\n"); return(2);
		}
		SerSend((out>>0)&0xFF);
	}

	// write back result
	*in=m;

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PicCmd(uint32 out,int32 *in)
{
	int32		i;

	// send cmd
	i=PicSwap(out,in);
	if(i) return(1);

	// get reply
	i=PicSwap(0x00000000,in);
	if(i) return(1);

	// cool
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int SerInit(void)
{
	unsigned short		*p;
	unsigned short far	*fp;

	// get address
	if(sizeof(fp)==6)
	{
		p=(unsigned short *)0x00000400;
		basePort=p[comPort-1];
	}
	else
	{
		fp=(unsigned short far *)0x00000400;
		basePort=fp[comPort-1];
	}
	if(!basePort) return(1);
	printf("using serial uart at %03x\n",basePort);

	// disable port interrupts
	outp(basePort+1,0);

	// disable FIFO mode
	outp(basePort+2,0);

	outp(basePort+4,0);

	// set 38400 baud, select 8-bit words
	outp(basePort+3,128);
	outp(basePort+0,0x03);
	outp(basePort+1,0x00);
	outp(basePort+3,3);		// 8-bits

	return(0);
}


/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SerRecv(void)
{
	int	j;

	if(!(inp(basePort+5)&1)) return(-1);
	j=inp(basePort);
//	printf("recv %d\n",j);
	return(j);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void SerSend(int32 ch)
{
	while(!(inp(basePort+5)&32));
//	printf("send %d\n",ch);
	outp(basePort,ch);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 PicInit(void)
{
	int32		i,j,k;
	unsigned short	*p;
	unsigned short far *fp;

	if(!comPort)
	{
		printf("resetting PIC..."); fflush(stdout);

		// lower MCLR
		ParWrite(MCLR_MASK,0);

		// wait a moment
		Delay(10);

		// raise MCLR
		ParWrite(MCLR_MASK,MCLR_MASK);

		printf("done\n");
		printf("waiting for PIC signal..."); fflush(stdout);
	}
	else
	{
		// send 0x09 (reset) until we get a '!'
		for(i=0;i<1000;i++)
		{
			printf("."); fflush(stdout);
			SerSend(0x09);
			k=-1;
			for(j=0;j<1000;j++)
			{
				k=SerRecv();
				if(k!=-1) break;
			}
			if(k=='!') break;
		}
		if(i==1000)
		{
			printf("can't start serial comms\n");
			return(1);
		}

		// wait for '*'
		while(SerRecv()!='*');

		// send nop
		SerSend(0);
		while(SerRecv()==-1);	// hi
		SerSend(0);
		while(SerRecv()==-1);	// lo
		SerSend(0);
		while(SerRecv()==-1);	// chk
		SerSend(0);
	}

	// try a NOP
	i=PicSwap(0x00000000,&j);
	if(i)
	{
		printf("failed on NOP\n");
		return(1);
	}

	// try an ECHO
	i=PicCmd(0x03001249,&j);
	if(j!=0x0249)
	{
		printf("failed on ECHO\n");
		return(i);
	}
	printf("ok\n");


	return(0);

}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void Delay(int32 p)
{
	int32		i;
	if(comPort)
		for(i=0;i<p*1000;i++) inp(basePort+5);
	else
		for(i=0;i<p*1000;i++) inp(basePort+1);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXCmd(int32 rept,int32 cmd,int32 data,int32 *result)
{
	uint32		q;
	int32		i,j;

	q =(uint32)0x04000000;
	q|=((uint32)rept)<<16;
	q|=((uint32)cmd)<<12;
	q|=((uint32)data) & 0xFFF;

	i=PicCmd(q,&j);
	if(i) return(-1);

	*result=j & 0xFFF;

	return((j>>12) & 0xF);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXInit(void)
{
	int32		i,j,k;

	printf("initialising SX..."); fflush(stdout);

	// lower OSC1
	i=PicCmd(0x01000000,&j);
	if(i) { printf("error\n"); return(1); }

	Delay(25);

	// lower OSC2
	i=PicCmd(0x02000000,&j);
	if(i) { printf("error\n"); return(1); }

	Delay(25);

	// toggle OSC1 from 0V to 5V [9 times]
	for(k=0;k<10;k++)
	{
		i=PicCmd(0x01000001,&j);
		if(i) { printf("error\n"); return(1); }
//		Delay(5);
		i=PicCmd(0x01000000,&j);
		if(i) { printf("error\n"); return(1); }
//		Delay(5);
	}

	// wait a moment... hopefully long enough for 9 internal clocks
	Delay(200);

	// release OSC2
	i=PicCmd(0x02000001,&j);
	if(i) { printf("error\n"); return(1); }

	Delay(100);

	// raise OSC1 to 12V
	i=PicCmd(0x01000002,&j);
	if(i) { printf("error\n"); return(1); }

	Delay(500);

	// see if the PIC can detect a 128KHz ISP signal
	for(k=0;k<10;k++)
	{
		printf("."); fflush(stdout);
		Delay(100);
		i=SXCmd(1,SX_NOP,0xFFF,&j);
		if(i<0) { printf("error\n"); return(1); }
		if(!i) break;
	}
	if(j!=0x0fff)
	{
		i=PicCmd(0x01000000,&j);
		return(2);
	}
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 SXClose(void)
{
	int32		i,j,k;

	// exit ISP mode
	printf("exiting ISP mode..."); fflush(stdout);
	i=PicCmd(0x01000000,&j);
	if(i) { printf("error\n"); return(1); }
	Delay(100);
	i=PicCmd(0x01000001,&j);
	if(i) { printf("error\n"); return(1); }
	Delay(100);
	i=PicCmd(0x01000000,&j);
	if(i) { printf("error\n"); return(1); }
	Delay(100);

	// ensure no ISP signal
	i=SXCmd(1,SX_NOP,0xFFF,&j);
	if(i!=3)
	{
		printf("SX still seems to be in ISP mode!\n");
		return(2);
	}

	printf("done\n");
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 HexFrom1(char s)
{
	int		i;

	i=toupper(s)-'0';
	if(i>10) i-=7;
	return(i);
}

int32 HexFrom2(char *s)
{
	return((HexFrom1(s[0])<<4) | HexFrom1(s[1]));
}

int32 HexFrom4(char *s)
{
	return((HexFrom2(&s[0])<<8) | HexFrom2(&s[2]));
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int32 HexRead(char *fn)
{
	FILE		*stream;
	int		i,j,k;
	int		bc,type,addr;

	// open file
	stream=fopen(fn,"r");
	if(!stream)
	{
		printf("can't open %s\n",fn);
		return(1);
	}

	// start with FFFF's everywhere
	for(i=0;i<PROGRAM_SIZE;i++)	progData[i]=0xFFFF;
	for(i=0;i<USER_SIZE;i++)	userData[i]=0xFFFF;
	for(i=0;i<EEPROM_SIZE;i++)	eepromData[i]=0xFFFF;

	// process a line at a time
	while(1)
	{
		// read line
		fgets(tmpString,1023,stream);
		if(feof(stream)) break;

		// get type, addr, length
		if(tmpString[0]!=':') continue;
		bc=HexFrom2(&tmpString[1]);
		addr=HexFrom4(&tmpString[3]);
		type=HexFrom2(&tmpString[7]);

		// check type
		if(type==1) break;
		if(type!=0)
		{
			printf("unrecognised HEX record type: $%02x\n",type);
			return(1);
		}

		bc/=2;
		addr/=2;
		for(i=0;i<bc;i++)
		{
			j=HexFrom4(&tmpString[i*4+9]);
			j=((j>>8)&0xFF) | (j<<8);

			if(!picMode)
			{
				j&=0xFFF;

				if(addr>=0 && addr<PROGRAM_SIZE)
				{
					progData[addr]=j;
				}
				else if(addr>=0x1000 && addr<0x1010)
				{
					// ignore
				}
				else if(addr==0x1010)
				{
					if(progFuse!=-1) printf("Warning: ignoring FUSE value in file %s\n",fn);
					else
					{
						progFuse=j;
					}
				}
				else if(addr==0x1011)
				{
					if(progFusex!=-1) printf("Warning: ignoring FUSEX value in file %s\n",fn);
					else
					{
						progFusex=j;

						// brownout hack
						if(progFusex & 0x020)	progFusex |= 0x010;
						else			progFusex &= ~0x010;
					}
				}
				else
				{
					printf("warning: data out of range in hex file: $%04x:$%02x\n",addr,j);
				}
			}
			else
			{
				j&=0xFFFF;
				if(addr>=0 && addr<PICPROGRAM_SIZE)
				{
					progData[addr]=j;
				}
				else if(addr>=0x2000 && addr<=0x2007)
				{
					userData[addr-0x2000]=j;
					if(addr==0x2007 && progFuse!=-1)
					{
						printf("Warning: ignoring FUSE value in file %s\n",fn);
					}
				}
				else if(addr>=0x2100 && addr<=0x213F)
				{
					eepromData[addr-0x2100]=j;
				}
				else
				{
					printf("warning: data out of range in hex file: $%04x:$%02x\n",addr,j);
				}
			}

			addr++;
		}
	}

	// close file
	fclose(stream);

	// did we get config word?
	if(picMode)
	{
		if(progFuse!=-1)	userData[0x0007]=progFuse;
		if(userData[0x0007]!=0xFFFF) progFuse=userData[0x0007];
	}

	// done
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int ReadINI(void)
{
	FILE		*stream;
	int		i,j,k;

	// set defaults
	writeRept=100;
	verify=0;

	stream=fopen("fluffy.ini","r");
	if(!stream) return(0);

	while(1)
	{
		i=fscanf(stream,"%s",tmpString);
		if(i!=1) break;

		if(!stricmp(tmpString,"verify"))
		{
			verify=1;
		}
		else if(!stricmp(tmpString,"com1")) comPort=1;
		else if(!stricmp(tmpString,"com2")) comPort=2;
		else if(!stricmp(tmpString,"com3")) comPort=3;
		else if(!stricmp(tmpString,"com4")) comPort=4;
		else if(!stricmp(tmpString,"picmode"))
		{
			picMode=1;
		}
		else if(!stricmp(tmpString,"writetime"))
		{
			i=fscanf(stream,"%d",&j);
			if(i!=1) { printf("Error in INI file - no valid writetime specified\n"); exit(1); }
			j=(j*32)/17;
			if(j<1) j=1;
			if(j>255) j=255;
			writeRept=j;
		}
		else if(!stricmp(tmpString,"fuse"))
		{
			j=fscanf(stream,"%lx",&progFuse);
			if(j!=1) { printf("Error in INI file - invalid hex fuse value\n"); exit(1); }
			i++;
		}
		else if(!stricmp(tmpString,"fusex"))
		{
			j=fscanf(stream,"%lx",&progFusex);
			if(j!=1) { printf("Error in INI file - invalid hex fusex value\n"); exit(1); }
			i++;
		}
	}

	fclose(stream);
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int ParseArgs(int argc,char **argv)
{
	int32		i,j,k;

	if(argc<2)
	{
		printf("usage: fluffy input.sxh [-verify]\n");
		printf("                        [-writetime n]  (in ms, default 50ms)\n");
		printf("                        [-fuse  FFF]    (in hex)\n");
		printf("                        [-fusex FFF]    (in hex)\n");
		printf("                        [-picmode]\n");
		printf("                        [-com1] [-com2] [-com3] [-com4]\n");
		printf("                        [-test]\n");
		printf(" or    fluffy -read\n");
		exit(1);
	}

	inputFile[0]=0;
	for(i=1;i<argc;i++)
	{
		if(argv[i][0]!='-')
		{
			if(inputFile[0]!=0)
			{
				printf("can't specify multiple input files!\n");
				return(1);
			}
			strcpy(inputFile,argv[i]);
		}
		else if(!stricmp(argv[i],"-verify"))
		{
			verify=1;
		}
		else if(!stricmp(argv[i],"-com1")) comPort=1;
		else if(!stricmp(argv[i],"-com2")) comPort=2;
		else if(!stricmp(argv[i],"-com3")) comPort=3;
		else if(!stricmp(argv[i],"-com4")) comPort=4;
		else if(!stricmp(argv[i],"-writetime"))
		{
			if(i==argc-1) { printf("must specify write count\n"); exit(1); }
			j=atoi(argv[i+1]);
			j=(j*32)/17;
			if(j<1) j=1;
			if(j>255) j=255;
			writeRept=j;
			i++;
		}
		else if(!stricmp(argv[i],"-forcefusex"))
		{
			// only use if you've inadvertantly screwed your fusex top bits!
			// and you know what they're supposed to be!
			forceFusex=1;
		}
		else if(!stricmp(argv[i],"-fuse"))
		{
			if(i==argc-1) { printf("must specify fuse value\n"); exit(1); }
			j=sscanf(argv[i+1],"%lx",&progFuse);
			if(j!=1) { printf("invalid hex fuse value\n"); exit(1); }
			i++;
		}
		else if(!stricmp(argv[i],"-fusex"))
		{
			if(i==argc-1) { printf("must specify fusex value\n"); exit(1); }
			j=sscanf(argv[i+1],"%lx",&progFusex);
			if(j!=1) { printf("invalid hex fusex value\n"); exit(1); }
			i++;
		}
		else if(!stricmp(argv[i],"-read"))
		{
			readMode=1;
		}
		else if(!stricmp(argv[i],"-test"))
		{
			testMode=1;
		}
		else if(!stricmp(argv[i],"-picmode"))
		{
			picMode=1;
		}
		else
		{
			printf("unrecognised parameter: %s\n",argv[i]);
			exit(1);
		}
	}
	if(!inputFile[0] && !readMode && !testMode)
	{
		printf("you must specify an input file\n");
		return(1);
	}

	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void SXProcess(void)
{
	int32		i,j,k,l;

	// init SX
	i=SXInit();
	if(i)
	{
		printf("Can't initialise SX\n");
		exit(1);
	}

	// get current device information
	printf("reading device information...\n");

	i=SXCmd(1,SX_READDEVICE,0xFFF,&sxDevice);
	if(i) {	printf("error\n"); SXClose(); exit(1); }

	i=SXCmd(1,SX_READFUSEX,0xFFF,&sxFusex);
	if(i) {	printf("error\n"); SXClose(); exit(1); }

	i=SXCmd(1,SX_READDATA,0xFFF,&sxFuse);
	if(i) {	printf("error\n"); SXClose(); exit(1); }

	printf("RAM             : %d banks\n",1<<((sxDevice>>2)&3));
	printf("Program Memory  : %d words\n",(1<<(sxDevice&3))*512);
	printf("\n");

	// do we just want to read it?
	if(readMode)
	{
		printf("Device : [%03lx]\n",sxDevice);
		printf("Fusex  : [%03lx]\n",sxFusex);
		printf("Fuse   : [%03lx]\n",sxFuse);
		printf("\n");
		for(i=0;i<PROGRAM_SIZE;i+=16)
		{
			printf("%04x : ",i);
			for(j=0;j<16;j++)
			{
				k=SXCmd(1,SX_INC,0xFFF,&l);
				if(k) {	printf("error\n"); SXClose(); exit(1); }
				k=SXCmd(1,SX_READDATA,0xFFF,&l);
				if(k) {	printf("error\n"); SXClose(); exit(1); }

				printf("%03x ",l);
			}
			printf("\n");
		}
		printf("\n\n");
		SXClose();
		exit(0);
	}

	// mask off preset bits
	if(!forceFusex)
	{
		progFusex = (progFusex & 0x07F) | (sxFusex & 0xF80);
	}

	// erase
	printf("Erasing..."); fflush(stdout);
	i=SXCmd(0,SX_ERASE,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
	printf("done\n");

	printf("Writing..."); fflush(stdout);

	// write FUSEX
	i=SXCmd(1,SX_LOADDATA,progFusex,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
	i=SXCmd(writeRept,SX_PROGFUSEX,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }

	// write FUSE
	i=SXCmd(1,SX_LOADDATA,progFuse,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
	i=SXCmd(writeRept,SX_PROGDATA,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }

	// move into program memory
	i=SXCmd(1,SX_INC,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }

	// write code...
	for(k=0;k<PROGRAM_SIZE;k++)
	{
		progData[k] &= 0xFFF;

		if(progData[k]!=0xFFF)
		{
			printf("."); fflush(stdout);
			i=SXCmd(1,SX_LOADDATA,progData[k],&j);
			if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
			i=SXCmd(writeRept,SX_PROGDATA,0xFFF,&j);
			if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
		}
		i=SXCmd(1,SX_INC,0xFFF,&j);
		if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
	}

	// done writing
	i=SXClose();

	if(!verify)
	{
		printf("Done.\n\n");
		exit(0);
	}

	// pause
	Delay(300);

	// verify?
	i=SXInit();
	if(i)
	{
		printf("Can't initialise SX\n");
		exit(1);
	}
	printf("\n");
	printf("Verifying...\n");

	// check fusex
	i=SXCmd(1,SX_READFUSEX,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
	if(j != progFusex) { printf("FUSEX verify error\n"); SXClose(); exit(1); }

	// check fuse
	i=SXCmd(1,SX_READDATA,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
	if(j != progFuse) { printf("FUSE verify error\n"); SXClose(); exit(1); }

	// move onto program memory
	i=SXCmd(1,SX_INC,0xFFF,&j);
	if(i) {	printf("error %d\n",i); SXClose(); exit(1); }

	// verify each location
	for(k=0;k<PROGRAM_SIZE;k++)
	{
		i=SXCmd(1,SX_READDATA,0xFFF,&j);
		if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
		if(j != progData[k]) { printf("Data verify error at $%03lx\n",k); SXClose(); exit(1); }

		i=SXCmd(1,SX_INC,0xFFF,&j);
		if(i) {	printf("error %d\n",i); SXClose(); exit(1); }
	}

	// yeehah
	printf("Verify OK\n\n");
	SXClose();
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int TPICInit(void)
{
	int32		i,j,k,cmd;

	// go to 0V
	cmd=0x01000000;
	i=PicCmd(cmd,&j);
	if(i || j!=0) return(1);

	// lower CLOCK/DATA
	cmd=0x05000000;
	i=PicCmd(cmd,&j);
	if(i || j!=0) return(1);

	// go to 12V
	cmd=0x01000002;
	i=PicCmd(cmd,&j);
	if(i || j!=0) return(1);

	// hopefully ok...
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

int TPICClose(void)
{
	int32		i,j,k,cmd;

	// go to 0V
	cmd=0x01000000;
	i=PicCmd(cmd,&j);
	if(i || j) return(1);

	// restore CLOCK/DATA to tristate
	cmd=0x08000000;
	i=PicCmd(cmd,&j);
	if(i || j!=0) return(2);

	// go to 5V
	cmd=0x01000001;
	i=PicCmd(cmd,&j);
	if(i || j!=0) return(3);

	// hopefully ok...
	return(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void PICProcess(void)
{
	uint32		cmd;
	int32		i,j,k,l;

	// init target chip
	printf("Initialising target PIC... "); fflush(stdout);
	i=TPICInit();
	if(i) { printf("error!\n"); TPICClose(); exit(1); }
	printf("ok\n");

	// read config
	printf("reading configuration word... "); fflush(stdout);
	i=PicCmd(0x0700FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
	for(k=0;k<7;k++)
	{
		i=PicCmd(0x0606FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
	}
	i=PicCmd(0x0704FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
	sxFuse=(j>>1) & 0x3fff;
	printf("[%04x]\n",sxFuse);
	if(!(sxFuse & 0x10))
	{
		if(readMode)
		{
			printf("Target PIC is code protected - unable to read\n");
			TPICClose();
			exit(1);
		}
		else
		{
			printf("Disabling code protection...\n");
			TPICClose(); TPICInit();
			i=PicCmd(0x0700FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
			for(k=0;k<7;k++)
			{
				i=PicCmd(0x0606FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
			}
			i=PicCmd(0x0601FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
			i=PicCmd(0x0607FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
			i=PicCmd(0x0608FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
			Delay(25);
			i=PicCmd(0x0607FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
			i=PicCmd(0x0608FFFF,&j); if(i) { printf("error!\n"); TPICClose(); exit(1); }
		}
	}

	// reset chip
	TPICClose(); TPICInit();

	// read if necessary
	if(readMode)
	{
		printf("\n\n");
		TPICClose(); TPICInit();
		printf("Program Memory:\n");
		for(i=0;i<PICPROGRAM_SIZE;i+=8)
		{
			printf("%04x : ",i);
			for(j=0;j<8;j++)
			{
				k=PicCmd(0x0704FFFF,&l);
				if(k) {	printf("error\n"); TPICClose(); exit(1); }
				l=(l>>1) & 0x3FFF;
				printf("%04lx ",l);
				k=PicCmd(0x0606FFFF,&l);
				if(k) {	printf("error\n"); TPICClose(); exit(1); }
			}
			printf("\n");
		}
		printf("\n\n");

		printf("\n\n");
		TPICClose(); TPICInit();
		printf("EEPROM Data Memory:\n");
		for(i=0;i<EEPROM_SIZE;i+=16)
		{
			printf("%04x : ",i);
			for(j=0;j<16;j++)
			{
				k=PicCmd(0x0705FFFF,&l);
				if(k) {	printf("error\n"); TPICClose(); exit(1); }
				l=(l>>1) & 0xFF;
				printf("%02lx ",l);
				k=PicCmd(0x0606FFFF,&l);
				if(k) {	printf("error\n"); TPICClose(); exit(1); }
			}
			printf("\n");
		}
		printf("\n\n");

		printf("\n\n");
		TPICClose(); TPICInit();
		PicCmd(0x0700FFFF,&l);
		printf("Configuration Memory:\n");
		for(i=0;i<USER_SIZE;i+=8)
		{
			printf("%04x : ",i);
			for(j=0;j<8;j++)
			{
				k=PicCmd(0x0704FFFF,&l);
				if(k) {	printf("error\n"); TPICClose(); exit(1); }
				l=(l>>1) & 0x3FFF;
				printf("%04lx ",l);
				k=PicCmd(0x0606FFFF,&l);
				if(k) {	printf("error\n"); TPICClose(); exit(1); }
			}
			printf("\n");
		}
		printf("\n\n");

		TPICClose();
		exit(0);
	}

	printf("writing..."); fflush(stdout);
	TPICClose(); TPICInit();
	for(i=0;i<PICPROGRAM_SIZE;i++)
	{
		progData[i] &= 0x3FFF;
		k=PicCmd(0x0704FFFF,&l); l=(l>>1) & 0x3fff;
		if(k) { printf("error\n"); TPICClose(); exit(1); }
		if(l!=progData[i])
		{
			j=PicCmd(0x07020000 | (progData[i]<<1),&l); if(j) { printf("error\n"); TPICClose(); exit(1); }
			j=PicCmd(0x0608ffff,&l); if(j) { printf("error\n"); TPICClose(); exit(1); }
			Delay(10);
			printf("."); fflush(stdout);
			if(verify)
			{
				k=PicCmd(0x0704FFFF,&l); l=(l>>1) & 0x3fff;
				if(k) { printf("error\n"); TPICClose(); exit(1); }
				if(l!=progData[i]) { printf("\nverify failure at [%04lx] %04lx:%04lx \n",i,progData[i],l); }
			}
		}
		k=PicCmd(0x0606FFFF,&l);
	}

	TPICClose(); TPICInit();
	for(i=0;i<EEPROM_SIZE;i++)
	{
		eepromData[i] &= 0xFF;
		k=PicCmd(0x0705FFFF,&l); l=(l>>1) & 0xff;
		if(k) { printf("error\n"); TPICClose(); exit(1); }
		if(l!=eepromData[i])
		{
			j=PicCmd(0x07030000 | (eepromData[i]<<1),&l); if(j) { printf("error\n"); TPICClose(); exit(1); }
			j=PicCmd(0x0608ffff,&l); if(j) { printf("error\n"); TPICClose(); exit(1); }
			Delay(10);
			printf("."); fflush(stdout);
			if(verify)
			{
				k=PicCmd(0x0705FFFF,&l); l=(l>>1) & 0xff;
				if(k) { printf("error\n"); TPICClose(); exit(1); }
				if(l!=eepromData[i]) { printf("\nverify failure at [%04lx] %02lx:%02lx \n",i+0x2100,eepromData[i],l); }
			}
		}
		k=PicCmd(0x0606FFFF,&l);
	}

	TPICClose(); TPICInit();
	PicCmd(0x0700FFFF,&l);
	for(i=0;i<USER_SIZE;i++)
	{
		userData[i] &= 0x3FFF;
		k=PicCmd(0x0704FFFF,&l); l=(l>>1) & 0x3fff;
		if(k) { printf("error\n"); TPICClose(); exit(1); }
		if(l!=userData[i])
		{
			j=PicCmd(0x07020000 | (userData[i]<<1),&l); if(j) { printf("error\n"); TPICClose(); exit(1); }
			j=PicCmd(0x0608ffff,&l); if(j) { printf("error\n"); TPICClose(); exit(1); }
			Delay(10);
			printf("."); fflush(stdout);
			if(verify)
			{
				k=PicCmd(0x0704FFFF,&l); l=(l>>1) & 0x3fff;
				if(k) { printf("error\n"); TPICClose(); exit(1); }
				if(l!=userData[i]) { printf("\nverify failure at [%04lx] %04lx:%04lx \n",i+0x2000,userData[i],l); }
			}
		}
		k=PicCmd(0x0606FFFF,&l);
	}
	printf("\n");

	printf("Resetting target PIC... "); fflush(stdout);
	i=TPICClose(); if(i) { printf("error! %d\n",i); exit(1); }
	printf("ok\n");




}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void TestProcess(void)
{
	int32		i,j,k;

	for(i=0;i<50;i++) printf("\n");
	printf("Fluffy Diagnostics\n");
	printf("------------------\n\n");

	printf("Ensure that the programmer is connected to ");
	if(comPort) printf("COM%d\n",comPort);
	else printf("the parallel port\n");
	printf("and the power supply is connected.\n");
	printf("\n");
	printf("When you have ensure that this is the case, press a key...\n");
	getch();

	printf("Initialising port... "); fflush(stdout);
	if(comPort)	SerInit();
	else		ParInit(888);
	printf("done\n");
	printf("Looking for programmer PIC... "); fflush(stdout);
	i=PICInit();
	if(i)
	{
		printf("failed\n");
		exit(1);
	}
	printf("done.\n");

	printf("Testing comms bits...\n");
	i=PicCmd(0x03001249,&j);
	if(j!=0x0249)
	{
		printf("failed on ECHO\n");
		exit(1);
	}
	printf("ok\n");

	printf("\n\n");
	printf("The next test will check that the programming voltage circuitry is\n");
	printf("working correctly. Ensure that the OSC1/OSC2/Vdd/SPARE/GND external\n");
	printf("connectors are left unconnected. Press a key when ready...\n\n");
	getch();
	printf("The voltage between Vdd and GND should be about 5V. If this is not the case,\n");
	printf("you have a problem! Press ESC to quit, or any other key to continue...\n\n");
	if(getch()==27) exit(1);
	printf("The voltage between OSC2 and GND should be about 5V. If this is not the case,\n");
	printf("you have a problem! Press ESC to quit, or any other key to continue...\n\n");
	if(getch()==27) exit(1);

	i=PicCmd(0x02000000,&j);
	if(i) { printf("communication error!\n"); exit(1); }
	printf("The voltage between OSC2 and GND should NOW be about 0V. If this is not the\n");
	printf("case, you have a problem! Press ESC to quit, or any other key to continue...\n\n");
	if(getch()==27) exit(1);
	i=PicCmd(0x02000001,&j);
	if(i) { printf("communication error!\n"); exit(1); }

	i=PicCmd(0x01000000,&j);
	if(i) { printf("communication error!\n"); exit(1); }
	printf("The voltage between GND and OSC1 should now be around 0V. If this is\n");
	printf("not the case, there is an error. Press ESC to quit, or any other key to\n");
	printf("continue...\n\n");
	if(getch()==27) exit(1);

	i=PicCmd(0x01000001,&j);
	if(i) { printf("communication error!\n"); exit(1); }
	printf("The voltage between GND and OSC1 should now be around 5V. If this is\n");
	printf("not the case, there is an error. Press ESC to quit, or any other key to\n");
	printf("continue...\n\n");
	if(getch()==27) exit(1);

	i=PicCmd(0x01000002,&j);
	if(i) { printf("communication error!\n"); exit(1); }
	printf("The voltage between GND and OSC1 should now be around 12.5V. If this is\n");
	printf("not the case, there is an error. Press ESC to quit, or any other key to\n");
	printf("continue...\n\n");
	if(getch()==27) exit(1);
	i=PicCmd(0x01000000,&j);

	printf("Ok... looking good so far. Now we try an SX chip...\n\n");
	printf("Connect up the SX's Vdd/GND, and OSC1/OSC2 inputs to the appropriate\n");
	printf("programmer outputs... Ensure the SX's MCLR input is pulled high.\n");
	printf("When everything is ready, press a key...\n");
	getch();

	i=SXInit();
	if(i)
	{
		printf("Can't detect SX chip.\n");
		exit(1);
	}

	printf("Detected SX chip ok... reading device word...\n\n");
	i=SXCmd(1,SX_READDEVICE,0xFFF,&sxDevice);
	if(i) {	printf("error\n"); SXClose(); exit(1); }

	printf("Device word: [%03x]\n",sxDevice);
	i=SXClose();
	if(i) {	printf("error closing device!!!\n"); SXClose(); exit(1); }

	printf("\n\n");
	printf("Congratulations - it would appear the your SX programmer\n");
	printf("is functioning correctly.\n\n");
	exit(0);
}

/***********************************************************************
 *
 *
 *
 ***********************************************************************
 *
 *
 *
 ***********************************************************************/

void main(int argc, char **argv)
{
	int32		i,j,k,l;

	// read INI file
	i=ReadINI();
	if(i) exit(1);

	// check cmd line args
	i=ParseArgs(argc,argv);
	if(i) exit(1);

	if(testMode)
	{
		TestProcess();
		exit(0);
	}

	// read hex file
	if(!readMode)
	{
		i=HexRead(inputFile);
		if(i) exit(1);

		// check FUSE and FUSEX valid
		if(progFuse==-1)
		{
			printf("Error: a valid value for FUSE was not found on the command line, or in\n");
			printf("       any of the input files.\n");
			exit(1);
		}
		if(progFusex==-1 && !picMode)
		{
			printf("Error: a valid value for FUSEX was not found on the command line, or in\n");
			printf("       any of the input files.\n");
			exit(1);
		}

		// display parameters
		printf("Parameters:\n");
		printf("   fuse       [%03lx]\n",progFuse);
		if(!picMode)
		{
			printf("   fusex      [%03lx]\n",progFusex);
			printf("   write rept [%d]\n",writeRept);
		}
		printf("   verify     [%s]\n",verify?"on":"off");
		printf("\n");
	}

	// init parallel port
	if(!comPort)	ParInit(888);
	else
	{
		printf("initting serial port\n");
		i=SerInit();
		if(i)
		{
			printf("Can't init serial port\n");
			exit(1);
		}
	}

	// init PIC
	i=PicInit();
	if(i)
	{
		printf("Error talking to PIC\n");
		exit(1);
	}

	if(!picMode)
	{
		SXProcess();
	}
	else
	{
		PICProcess();
	}
}


